document.addEventListener('DOMContentLoaded', () => {
  const player = videojs('videojs-player', {
    controls: false,
    autoplay: true,
    preload: 'auto',
    fluid: true
  });

  const controlsContainer = document.querySelector('.controls-container');
  const playPauseButton = document.querySelector('.controls button.play-pause');
  const volumeButton = document.querySelector('.controls button.volume');
  const fullScreenButton = document.querySelector('.controls button.full-screen');
  const episodesButton = document.querySelector('.controls button.episodes'); // Botón de episodios
  const episodesMenu = document.querySelector('.episodes-menu'); // Menú de episodios
  const castButton = document.querySelector('.controls button.cast'); // Botón Cast (Picture-in-Picture)

  const playButton = playPauseButton.querySelector('.playing');
  const pauseButton = playPauseButton.querySelector('.paused');
  const fullVolumeButton = volumeButton.querySelector('.full-volume');
  const mutedButton = volumeButton.querySelector('.muted');

  const progressBar = document.querySelector('.progress-bar');
  const watchedBar = document.querySelector('.watched-bar');
  const timeLeft = document.querySelector('.time-remaining');
  const sideMenu = document.querySelector('.side-menu');
  const channelInfoBox = document.querySelector('.channel-info-box');
  const channelTitle = document.querySelector('.channel-title');

  let controlsTimeout;
  controlsContainer.style.opacity = '0';
  watchedBar.style.width = '0px';
  pauseButton.style.display = 'none';

  // Ocultar el menú de episodios al inicio
  if (episodesMenu) {
    episodesMenu.style.display = 'none';
  }

  // Función para actualizar el botón de Play/Pause
  function updatePlayPauseButton() {
    if (player.paused()) {
      playButton.style.display = '';
      pauseButton.style.display = 'none';
    } else {
      playButton.style.display = 'none';
      pauseButton.style.display = '';
    }
  }

  player.on('play', updatePlayPauseButton);
  player.on('pause', updatePlayPauseButton);

  // Mostrar controles cuando se cambia de canal
  player.on('loadeddata', () => {
    displayControls();
    updatePlayPauseButton();
  });

  // Función para mostrar/ocultar controles
  function displayControls() {
    controlsContainer.style.opacity = '1';
    document.body.style.cursor = 'initial';

    if (controlsTimeout) {
      clearTimeout(controlsTimeout);
    }
    controlsTimeout = setTimeout(() => {
      controlsContainer.style.opacity = '0';
      document.body.style.cursor = 'none';
    }, 5000);
  }

  // Función Play/Pause
  function playPause() {
    if (player.paused()) {
      player.play();
    } else {
      player.pause();
    }
  }

  // Función para activar/desactivar el volumen
  function toggleMute() {
    player.muted(!player.muted());

    if (player.muted()) {
      fullVolumeButton.style.display = 'none';
      mutedButton.style.display = '';
    } else {
      fullVolumeButton.style.display = '';
      mutedButton.style.display = 'none';
    }
  }

  // Click en la barra de progreso para saltar a una posición determinada
  progressBar.addEventListener('click', (event) => {
    const rect = progressBar.getBoundingClientRect();
    const offsetX = event.clientX - rect.left;
    const pos = offsetX / progressBar.offsetWidth;
    player.currentTime(pos * player.duration());
  });

  // Eventos de los botones
  playPauseButton.addEventListener('click', playPause);
  volumeButton.addEventListener('click', toggleMute);
  fullScreenButton.addEventListener('click', () => {
    if (player.isFullscreen()) {
      player.exitFullscreen();
    } else {
      player.requestFullscreen();
    }
  });

  // Mostrar controles al mover el mouse
  document.addEventListener('mousemove', displayControls);

  // Atajos de teclado
  document.addEventListener('keyup', (event) => {
    if (event.code === 'Space') {
      playPause();
    } else if (event.code === 'KeyM') {
      toggleMute();
    }
    displayControls();
  });

  // Función para cambiar de canal
  function cambiarCanal(nuevaURL, nuevoNombreDelCanal) {
    // Ocultar el menú lateral al cambiar de canal
    if (sideMenu) {
      sideMenu.style.display = 'none';
    }

    // Ocultar el cuadro de información del canal
    if (channelInfoBox) {
      channelInfoBox.style.display = 'none';
    }

    // Actualizar el título del canal si existe
    if (channelTitle) {
      channelTitle.textContent = nuevoNombreDelCanal;
    }

    // Cambiar la fuente del video y recargarlo
    player.src({ src: nuevaURL, type: 'application/x-mpegURL' });
    player.load();
    player.play();

    // Asegurar que el botón Play/Pause se actualice correctamente
    updatePlayPauseButton();
  }

  // ***** Código para mostrar/ocultar el menú de episodios *****
  if (episodesButton && episodesMenu) {
    episodesButton.addEventListener('click', () => {
      episodesMenu.style.display = episodesMenu.style.display === 'none' ? 'block' : 'none';
    });
  }

  // ***** Función para activar/desactivar Picture-in-Picture *****
  async function togglePictureInPicture() {
    try {
      const videoEl = player.tech(true).el();
      if (document.pictureInPictureElement) {
        await document.exitPictureInPicture();
      } else {
        await videoEl.requestPictureInPicture();
      }
    } catch (error) {
      console.error('Error al alternar Picture-in-Picture:', error);
    }
  }

  // Evento del botón Cast (Picture-in-Picture)
  if (castButton) {
    castButton.addEventListener('click', togglePictureInPicture);
  }
});
